/*
 * Decompiled with CFR 0.152.
 */
package db.manager.files;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.log4j.Logger;

public class FileUtil {
    private static final Logger LOGGER = Logger.getLogger(FileUtil.class);
    private static final int ONE_KB = 1000;

    public static String getSizeDisplay(long numberOfBytes) {
        Object result;
        if (numberOfBytes < 1000L) {
            result = numberOfBytes + " B";
        } else {
            int exp = (int)(Math.log(numberOfBytes) / Math.log(1000.0));
            Character pre = Character.valueOf("KMGTPEZY".charAt(exp - 1));
            result = String.format("%.1f %sB", (double)numberOfBytes / Math.pow(1000.0, exp), pre);
        }
        return result;
    }

    public static String toUnixStyle(String path) {
        return path == null ? null : path.replace('\\', '/');
    }

    public static String readFile(InputStream is) {
        return new Scanner(is).useDelimiter("\\A").next();
    }

    public static List<String> readFileByLine(InputStream is) {
        ArrayList<String> lines = new ArrayList<String>();
        Scanner s = new Scanner(is);
        while (s.hasNext()) {
            lines.add(s.nextLine());
        }
        return lines;
    }

    private static SimpleFileVisitor<Path> visiteAndDeleteRun(final boolean deleteRoot, final boolean dryRun) {
        return new SimpleFileVisitor<Path>(){
            private int depth = 0;

            private boolean deleteItem() {
                return this.depth != 0 || deleteRoot;
            }

            private String getLogMessage(Path item, String itemType) {
                String actionPerformed = this.deleteItem() ? "Delete" : "Keep";
                return String.format("(%d) %s: %s => %s", this.depth, itemType, item.toFile().getAbsolutePath(), actionPerformed);
            }

            private void log(Path item, String itemType) {
                if (dryRun) {
                    LOGGER.info((Object)this.getLogMessage(item, itemType));
                } else if (LOGGER.isTraceEnabled()) {
                    String msg = this.getLogMessage(item, itemType);
                    LOGGER.trace((Object)msg);
                }
            }

            private void deleteItem(Path item) throws IOException {
                if (!dryRun && this.deleteItem()) {
                    Files.delete(item);
                }
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                this.log(file, "file");
                this.deleteItem(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException ioe) {
                return this.handleException(ioe);
            }

            private FileVisitResult handleException(IOException ioe) {
                LOGGER.error((Object)"deleteFileOrFolder : Aborting, caught unexpected exception", (Throwable)ioe);
                return FileVisitResult.TERMINATE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                ++this.depth;
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException ioe) throws IOException {
                if (ioe != null) {
                    return this.handleException(ioe);
                }
                --this.depth;
                this.log(dir, "folder");
                this.deleteItem(dir);
                return FileVisitResult.CONTINUE;
            }
        };
    }

    public static void removeDirectory(Path path, boolean deleteRoot, boolean dryRun) throws IOException {
        Files.walkFileTree(path, FileUtil.visiteAndDeleteRun(deleteRoot, dryRun));
    }

    public static void removeDirectory(Path path, boolean deleteRoot) throws IOException {
        FileUtil.removeDirectory(path, deleteRoot, false);
    }

    public static long sizeOfFileOrFolder(Path path) {
        final AtomicLong size = new AtomicLong(0L);
        try {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.addAndGet(attrs.size());
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException ioe) {
                    LOGGER.warn((Object)String.format("size : skipped file (%s) because of the following exception", file), (Throwable)ioe);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException ioe) {
                    if (ioe != null) {
                        LOGGER.warn((Object)String.format("size : had trouble traversing (%s) because of the following exception", dir), (Throwable)ioe);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException ioe) {
            LOGGER.error((Object)"size : Caught unexpected exception", (Throwable)ioe);
            throw new AssertionError((Object)"walkFileTree will not throw IOException if the FileVisitor does not");
        }
        return size.get();
    }

    public static String sizeDisplayOfFileOrFolder(Path path) {
        return FileUtil.getSizeDisplay(FileUtil.sizeOfFileOrFolder(path));
    }

    public static Boolean fileExists(Path p) {
        LOGGER.debug((Object)String.format("fileExistss %s", p.toString()));
        boolean test1 = Files.exists(p, new LinkOption[0]);
        boolean test2 = Files.notExists(p, new LinkOption[0]);
        Boolean result = !test1 && !test2 ? null : Boolean.valueOf(test1);
        LOGGER.debug((Object)String.format("Files.exists : %s, Files.notExists : %s, result : %s", Boolean.toString(test1), Boolean.toString(test2), result == null ? "Unknown" : Boolean.toString(result)));
        return result;
    }

    public static Boolean fileExists(File f) {
        return FileUtil.fileExists(f.toPath());
    }

    private FileUtil() {
    }
}

