/*
 * Decompiled with CFR 0.152.
 */
package oracle.dbtools.plusplus.connections.db.storage;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclEntryType;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Stream;
import oracle.dbtools.common.utils.PlatformUtils;
import oracle.dbtools.plusplus.connections.db.storage.NamedConnectionDefinition;
import oracle.dbtools.plusplus.connections.db.storage.StoreException;
import oracle.dbtools.util.InputOutputStreams;
import oracle.dbtools.util.Logger;

public final class ConnectionStorage {
    private static final String STORE_DIRECTORY_NAME = "connections";
    private final Path store;
    private static final Set<AclEntryPermission> ACL_FOLDER_PERMISSIONS = new LinkedHashSet<AclEntryPermission>(Arrays.asList(AclEntryPermission.values()));
    private static final Set<AclEntryPermission> ACL_MINIMUM_PERMISSIONS = EnumSet.of(AclEntryPermission.READ_DATA);
    private static final EnumSet<PosixFilePermission> POSIX_SECURE_PERMISSIONS = EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE);
    private static final FileAttribute<Set<PosixFilePermission>> POSIX_FOLDER_PERMISSIONS = PosixFilePermissions.asFileAttribute(POSIX_SECURE_PERMISSIONS);

    public static ConnectionStorage instance() {
        return Holder.INSTANCE;
    }

    private ConnectionStorage(Path store) {
        this.store = store;
    }

    public List<String> listConnections() {
        ArrayList<String> conns = new ArrayList<String>();
        if (this.store != null) {
            try (Stream<Path> dirs = Files.find(this.store, 1, (p, a) -> a.isDirectory(), new FileVisitOption[0]);){
                dirs.filter(ConnectionStorage::hasPropertiesFile).filter(ConnectionStorage::validatePermissions).forEach(dir -> conns.add(dir.getFileName().toString()));
            }
            catch (IOException ex) {
                Logger.severe(ConnectionStorage.class, "Error processing connections directory " + this.store, ex);
            }
        }
        return conns;
    }

    private static boolean hasPropertiesFile(Path dir) {
        return Files.exists(dir.resolve("dbtools.properties"), new LinkOption[0]);
    }

    private static boolean validatePermissions(Path dir) {
        try {
            if (!ConnectionStorage.isSecure(dir)) {
                Logger.severe(ConnectionStorage.class, "Incorrect file permissions for " + dir.toString());
            }
            return true;
        }
        catch (IOException ex) {
            Logger.severe(ConnectionStorage.class, "Error checking file permissions for " + dir.toString());
            return false;
        }
    }

    public NamedConnectionDefinition loadConnection(String identifier) {
        NamedConnectionDefinition def = null;
        Path dir = this.store.resolve(identifier);
        if (ConnectionStorage.validatePermissions(dir)) {
            def = NamedConnectionDefinition.createConnectionDefinition(dir);
        }
        return def;
    }

    Path duplicateConnection(String identifier, String newIdentifier, CopyErrorHandler errorHandler) throws IOException {
        Path origin = this.store.resolve(identifier);
        Path dest = this.store.resolve(newIdentifier);
        if (Files.exists(dest, new LinkOption[0])) {
            throw new StoreException("Destination connection directory exists");
        }
        ConnectionStorage.verifyAndCreateDirectory(dest);
        try (Stream<Path> s = Files.walk(origin, new FileVisitOption[0]);){
            s.forEach(p -> {
                Path rel = origin.relativize((Path)p);
                Path destFile = dest.resolve(rel);
                try {
                    if (Files.isDirectory(p, new LinkOption[0])) {
                        ConnectionStorage.verifyAndCreateDirectory(destFile);
                    } else {
                        Files.copy(p, destFile, StandardCopyOption.COPY_ATTRIBUTES);
                    }
                }
                catch (IOException ex) {
                    errorHandler.handleCopyError(ex);
                }
            });
        }
        return dest;
    }

    public OutputStream resolve(Path relativePath) throws IOException {
        Path resolvedPath = this.store.resolve(relativePath);
        Path connDir = resolvedPath.getParent();
        ConnectionStorage.verifyAndCreateDirectory(connDir);
        return Files.newOutputStream(resolvedPath, new OpenOption[0]);
    }

    public <T> void storeObject(String identifier, String filename, T data, StoreAction<T> action) throws StoreException {
        Path conndir = this.store.resolve(identifier);
        ConnectionStorage.verifyAndCreateDirectory(conndir);
        try (OutputStream out = Files.newOutputStream(conndir.resolve(filename), new OpenOption[0]);){
            action.store(data, out);
        }
        catch (IOException ex) {
            throw new StoreException("Error storing properties file " + filename, ex);
        }
    }

    public void storeDefinition(NamedConnectionDefinition def) throws StoreException {
        def.write(this);
    }

    public void createSubdirectory(String identifier, Path dirname) throws IOException {
        Path conndir = this.store.resolve(identifier);
        ConnectionStorage.verifyAndCreateDirectory(conndir);
        Path newDir = conndir.resolve(dirname);
        ConnectionStorage.verifyAndCreateDirectory(newDir);
    }

    private static Path getConnectionStorageDirectory() throws StoreException {
        Path root = PlatformUtils.getSharedConfigDirectory();
        Path store = root.resolve(STORE_DIRECTORY_NAME);
        ConnectionStorage.verifyAndCreateDirectory(store);
        return store;
    }

    private static void verifyAndCreateDirectory(Path path) throws StoreException {
        if (!Files.exists(path, new LinkOption[0])) {
            try {
                ConnectionStorage.createDirectories(path);
            }
            catch (IOException e) {
                throw new StoreException("TODO: Error creating connections dir", e);
            }
        } else if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new StoreException("TODO: The path " + path + " is not a directory");
        }
    }

    static Properties loadProperties(Path file) {
        Properties props = new Properties();
        Logger.finer(NamedConnectionDefinition.class, "Loading file " + file.toString());
        if (Files.isRegularFile(file, new LinkOption[0])) {
            try (InputStream is = Files.newInputStream(file, new OpenOption[0]);){
                props.load(InputOutputStreams.instance().reader(is));
            }
            catch (IOException ex) {
                Logger.warn(NamedConnectionDefinition.class, "Unexpected error reading " + file, ex);
            }
        }
        return props;
    }

    public static boolean isSecure(Path folder) throws IOException {
        if (ConnectionStorage.supportsPosix(folder)) {
            return ConnectionStorage.isSecurePosix(folder);
        }
        if (ConnectionStorage.supportsAcls(folder)) {
            return ConnectionStorage.isSecureAcl(folder);
        }
        return false;
    }

    public static void createDirectories(Path folder) throws IOException {
        if (ConnectionStorage.supportsPosix(folder)) {
            ConnectionStorage.createDirectoriesWithPosix(folder);
        } else if (ConnectionStorage.supportsAcls(folder)) {
            ConnectionStorage.createDirectoriesWithAcl(folder);
        } else {
            throw new IllegalArgumentException("Filesystem lacks capability to configure file permissions");
        }
    }

    private static void createDirectoriesWithAcl(Path folder) throws IOException {
        UserPrincipal userPrincipal = folder.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(System.getProperty("user.name"));
        AclEntry aclEntry = AclEntry.newBuilder().setPrincipal(userPrincipal).setPermissions(ACL_FOLDER_PERMISSIONS).setType(AclEntryType.ALLOW).build();
        Files.createDirectories(folder, new FileAttribute[0]);
        AclFileAttributeView aclView = Files.getFileAttributeView(folder, AclFileAttributeView.class, new LinkOption[0]);
        aclView.setAcl(Collections.singletonList(aclEntry));
    }

    private static boolean isSecureAcl(Path folder) throws IOException {
        UserPrincipal userPrincipal = folder.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(System.getProperty("user.name"));
        AclFileAttributeView aclView = Files.getFileAttributeView(folder, AclFileAttributeView.class, new LinkOption[0]);
        List<AclEntry> acls = aclView.getAcl();
        if (acls.isEmpty()) {
            return false;
        }
        for (AclEntry acl : acls) {
            if (ConnectionStorage.isSecureAclEntry(userPrincipal, acl)) continue;
            return false;
        }
        return true;
    }

    private static boolean isSecureAclEntry(UserPrincipal userPrincipal, AclEntry acl) {
        return userPrincipal.equals(acl.principal()) && acl.permissions().containsAll(ACL_MINIMUM_PERMISSIONS) && acl.type() == AclEntryType.ALLOW;
    }

    private static boolean supportsAcls(Path folder) {
        return folder.getFileSystem().supportedFileAttributeViews().contains("acl");
    }

    private static void createDirectoriesWithPosix(Path folder) throws IOException {
        Files.createDirectories(folder, POSIX_FOLDER_PERMISSIONS);
    }

    private static boolean isSecurePosix(Path folder) throws IOException {
        if (Files.exists(folder, new LinkOption[0]) && Files.isDirectory(folder, new LinkOption[0])) {
            Set<PosixFilePermission> actual = Files.getPosixFilePermissions(folder, new LinkOption[0]);
            return POSIX_SECURE_PERMISSIONS.equals(actual);
        }
        return false;
    }

    private static boolean supportsPosix(Path folder) {
        return folder.getFileSystem().supportedFileAttributeViews().contains("posix");
    }

    public static interface StoreAction<T> {
        public void store(T var1, OutputStream var2) throws IOException;
    }

    public static interface CopySupport {
        public OutputStream open(Path var1) throws IOException;
    }

    static interface CopyErrorHandler {
        public void handleCopyError(IOException var1);
    }

    public static final class Holder {
        private static final ConnectionStorage INSTANCE;

        static {
            Path store;
            try {
                store = ConnectionStorage.getConnectionStorageDirectory();
            }
            catch (StoreException e) {
                throw new RuntimeException(e);
            }
            INSTANCE = new ConnectionStorage(store);
        }
    }
}

