/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.cie.gdr.installation;

import com.oracle.cie.common.util.StringUtil;
import com.oracle.cie.gdr.nio.NioFileLockService;
import com.oracle.cie.gdr.nio.NioHelper;
import com.oracle.cie.gdr.utils.GdrException;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ReferenceCounter {
    private static final Logger _log = Logger.getLogger(ReferenceCounter.class.getName());
    public static final String REFERENCE_COUNT_PATH = "inventory/refcounts";
    public static final String REFERENCE_FILE_EXTENSION = ".ref";
    private static Map<Path, ReferenceCounter> _referenceCounters = new HashMap<Path, ReferenceCounter>();
    private static Map<Path, Object> _lockObjects = new HashMap<Path, Object>();
    private Path _filePath;
    private int _refCount = 0;
    private boolean _invalidated;

    public static ReferenceCounter getReferenceCounter(File home, RefType type, String name, String version) throws GdrException {
        return ReferenceCounter.getReferenceCounter(NioHelper.getInstance().getPath(home), type, name, version);
    }

    public static ReferenceCounter getReferenceCounter(Path home, RefType type, String name, String version) throws GdrException {
        return ReferenceCounter.getReferenceCounter(home, type, name, version, null, true);
    }

    public static ReferenceCounter getReferenceCounter(Path home, RefType type, String name, String version, String qualifier) throws GdrException {
        return ReferenceCounter.getReferenceCounter(home, type, name, version, qualifier, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static ReferenceCounter getReferenceCounter(Path home, RefType type, String name, String version, String qualifier, boolean wait) throws GdrException {
        if (_log.isLoggable(Level.FINER)) {
            _log.finer("Attempting to get reference counter for home: " + home + " type: " + (Object)((Object)type) + " name: " + name + " version: " + version + " qualifier: " + qualifier);
        }
        if (home == null) {
            GdrException ex = new GdrException("The home directory provided was null.");
            _log.throwing(ReferenceCounter.class.getName(), "ReferenceCounter", ex);
            throw ex;
        }
        if (type == null) {
            GdrException ex = new GdrException("The reference type provided was null.");
            _log.throwing(ReferenceCounter.class.getName(), "ReferenceCounter", ex);
            throw ex;
        }
        if (name == null) {
            GdrException ex = new GdrException("The reference name provided was null.");
            _log.throwing(ReferenceCounter.class.getName(), "ReferenceCounter", ex);
            throw ex;
        }
        if (version == null) {
            GdrException ex = new GdrException("The reference version provided was null.");
            _log.throwing(ReferenceCounter.class.getName(), "ReferenceCounter", ex);
            throw ex;
        }
        Path file = ReferenceCounter.createReferenceCountFile(home, type, name, version, qualifier);
        Object object = _lockObjects.get(file);
        synchronized (object) {
            ReferenceCounter refCounter = _referenceCounters.get(file);
            if (refCounter != null) {
                if (wait) {
                    while (_referenceCounters.containsKey(file)) {
                        try {
                            Thread.sleep(100L);
                        }
                        catch (InterruptedException interruptedException) {}
                    }
                } else {
                    return null;
                }
            }
            return ReferenceCounter.createReferenceCounter(file, wait);
        }
    }

    public static void releaseReferenceCounter(ReferenceCounter refCounter) throws GdrException {
        ReferenceCounter.releaseReferenceCounter(refCounter.getFilePath());
    }

    public static void releaseReferenceCounter(Path file) throws GdrException {
        ReferenceCounter refCounter = _referenceCounters.get(file);
        if (refCounter != null) {
            refCounter.endOperation();
        }
        _referenceCounters.remove(file);
    }

    private static ReferenceCounter createReferenceCounter(Path path, boolean wait) throws GdrException {
        _referenceCounters.put(path, new ReferenceCounter(path));
        ReferenceCounter refCounter = _referenceCounters.get(path);
        if (!refCounter.startOperation(wait)) {
            _referenceCounters.remove(path);
            return null;
        }
        return refCounter;
    }

    private static Path createReferenceCountFile(Path home, RefType type, String name, String version, String qualifier) {
        Path path = ReferenceCounter.getReferenceCountDirectory(home, type).resolve(ReferenceCounter.getReferenceCountFileName(name, version, qualifier));
        if (!_lockObjects.containsKey(path)) {
            _lockObjects.put(path, new Object());
        }
        return path;
    }

    public static Path getReferenceCountDirectory(Path home, RefType type) {
        return home.resolve(REFERENCE_COUNT_PATH).resolve(type.name());
    }

    public static String getReferenceCountFileName(String name, String version, String qualifier) {
        String filename = name.replaceAll(" ", "_") + "_" + version;
        if (!StringUtil.isNullOrEmpty((String)qualifier)) {
            filename = filename + "_" + qualifier;
        }
        return filename + REFERENCE_FILE_EXTENSION;
    }

    public static int getReferenceCount(File home, RefType type, String name, String version) {
        return ReferenceCounter.getReferenceCount(home, type, name, version, null);
    }

    public static int getReferenceCount(Path home, RefType type, String name, String version) {
        return ReferenceCounter.getReferenceCount(home, type, name, version, null);
    }

    public static int getReferenceCount(File home, RefType type, String name, String version, String qualifier) {
        return ReferenceCounter.getReferenceCount(NioHelper.getInstance().getPath(home), type, name, version, qualifier);
    }

    public static int getReferenceCount(Path home, RefType type, String name, String version, String qualifier) {
        int refCount = 0;
        Path file = ReferenceCounter.createReferenceCountFile(home, type, name, version, qualifier);
        if (Files.isRegularFile(file, new LinkOption[0])) {
            try {
                refCount = ReferenceCounter.getCurrentReferenceCount(file);
                if (!_referenceCounters.containsKey(file)) {
                    NioFileLockService.closeFileChannel((Path)file);
                }
            }
            catch (Exception e) {
                _log.fine("Error getting reference count -- " + e.getMessage());
            }
        }
        return refCount;
    }

    private ReferenceCounter(Path file) throws GdrException {
        if (file == null) {
            GdrException ex = new GdrException("The home directory provided was null.");
            _log.throwing(ReferenceCounter.class.getName(), "ReferenceCounter", ex);
            throw ex;
        }
        this._filePath = file;
    }

    public Path getFilePath() {
        return this._filePath;
    }

    public int getRefCount() throws GdrException {
        this.checkInvalidated();
        return this._refCount;
    }

    public boolean isInvalidated() {
        return this._invalidated;
    }

    private boolean startOperation(boolean wait) throws GdrException {
        Path file = this.getFilePath();
        if (_log.isLoggable(Level.FINER)) {
            _log.finer("Attempting to start reference count operation for file: " + file);
        }
        try {
            if (Files.isDirectory(file, new LinkOption[0])) {
                GdrException ex = new GdrException("Unable to obtain lock since reference count file was a directory: " + file);
                _log.throwing(ReferenceCounter.class.getName(), "startOperation", ex);
                throw ex;
            }
            if (!NioFileLockService.getFileLock((Path)file, (boolean)wait)) {
                return false;
            }
            this._refCount = ReferenceCounter.getCurrentReferenceCount(this.getFilePath());
            if (_log.isLoggable(Level.FINER)) {
                _log.finer("Successfully started reference count operation for file: " + file);
            }
        }
        catch (IOException e) {
            throw new GdrException((Throwable)e);
        }
        return true;
    }

    private void endOperation() throws GdrException {
        this.checkInvalidated();
        Path file = this.getFilePath();
        if (_log.isLoggable(Level.FINER)) {
            _log.finer("Attempting to end operation for file: " + file);
        }
        try {
            FileChannel fileChannel = NioFileLockService.getFileChannel((Path)file);
            fileChannel.position(0L);
            String refCountString = "" + this._refCount;
            ByteBuffer buffer = ByteBuffer.wrap(refCountString.getBytes());
            fileChannel.write(buffer);
            NioFileLockService.releaseFileLock((Path)file);
            if (this._refCount < 1) {
                if (_log.isLoggable(Level.FINER)) {
                    _log.finer("Refcount 0, deleting ref count file: " + file);
                }
                Files.delete(file);
                Path parent = file.getParent();
                try {
                    Files.delete(parent);
                    Files.delete(parent.getParent());
                }
                catch (DirectoryNotEmptyException e) {
                    _log.finer("Unable to delete directory " + parent);
                }
            }
            if (_log.isLoggable(Level.FINER)) {
                _log.finer("Successfully ended reference count operation for file: " + file);
            }
        }
        catch (IOException e) {
            throw new GdrException((Throwable)e);
        }
        this._invalidated = true;
    }

    public void increment() throws GdrException {
        this.checkInvalidated();
        if (_log.isLoggable(Level.FINER)) {
            _log.finer("Incrementing reference count for file: " + this.getFilePath() + " current value=" + this._refCount);
        }
        ++this._refCount;
    }

    public void decrement() throws GdrException {
        this.checkInvalidated();
        if (this._refCount > 0) {
            if (_log.isLoggable(Level.FINER)) {
                _log.finer("Decrementing reference count for file: " + this.getFilePath() + " current value=" + this._refCount);
            }
            --this._refCount;
        }
    }

    public void setRefCount(int refCount) throws GdrException {
        this.checkInvalidated();
        this._refCount = refCount;
    }

    private void checkInvalidated() throws GdrException {
        if (this.isInvalidated()) {
            GdrException ex = new GdrException("This ReferenceCounter object has been invalidated and may no longer be used.");
            _log.throwing(ReferenceCounter.class.getName(), "checkInvalidated", ex);
            throw ex;
        }
    }

    private static int getCurrentReferenceCount(Path filePath) throws GdrException {
        int refCount;
        try {
            if (Files.isRegularFile(filePath, new LinkOption[0])) {
                ByteBuffer buffer;
                FileChannel fileChannel;
                int bytes;
                if (_log.isLoggable(Level.FINER)) {
                    _log.finer("Setting current reference count for file: " + filePath);
                }
                if ((bytes = (fileChannel = NioFileLockService.getFileChannel((Path)filePath)).read(buffer = ByteBuffer.allocate((int)fileChannel.size()))) > 0) {
                    try {
                        String s = "";
                        buffer.flip();
                        for (int i = 0; i < bytes; ++i) {
                            s = s + (char)buffer.get();
                        }
                        refCount = Integer.parseInt(s.trim());
                    }
                    catch (NumberFormatException nfe) {
                        GdrException ex = new GdrException("Encountered invalid reference count value in file: " + filePath, (Throwable)nfe);
                        _log.throwing(ReferenceCounter.class.getName(), "getCurrentReferenceCount", ex);
                        throw ex;
                    }
                } else {
                    if (_log.isLoggable(Level.FINER)) {
                        _log.finer("Reference count file was empty for file:" + filePath + " defaulting to 0");
                    }
                    refCount = 0;
                }
                if (_log.isLoggable(Level.FINER)) {
                    _log.finer("Current reference count for file: " + filePath + " is " + refCount);
                }
                if (refCount < 0) {
                    GdrException ex = new GdrException("Encountered invalid reference count value \"" + refCount + "\" in file: " + filePath);
                    _log.throwing(ReferenceCounter.class.getName(), "getCurrentReferenceCount", ex);
                    throw ex;
                }
            } else {
                if (_log.isLoggable(Level.FINER)) {
                    _log.finer("Reference count file did not exist for file:" + filePath + " defaulting to 0");
                }
                refCount = 0;
            }
        }
        catch (IOException e) {
            throw new GdrException((Throwable)e);
        }
        return refCount;
    }

    public String toString() {
        return String.valueOf(this._refCount);
    }

    public static enum RefType {
        featuresets,
        components;

    }
}

